from agent.local_planner import LocalPlanner, RoadOption
from srunner.scenariomanager.carla_data_provider import CarlaDataProvider
from agent.utils.scenario_helper import generate_target_waypoint_list_multilane

COMMANDS = ['Lane Follow',
            'Opposite Lane Follow',
            'Speed Up',
            'Slow Down',
            'Change to Left Lane',
            'Change to Right Lane',
            'Stop',
            'Turn Left',
            'Turn Right',
            ]

class AtomicCommandFollower:
    """
    Atomic command follower implements basic commands for the vehicle to follow that 
    can be called by the LLMs to execute complex maneuvers.
    """
    def __init__(self, vehicle):
        """
        :param vehicle: actor to apply to local planner logic onto
        :param opt_dict: dictionary of local planner parameters
            dt: time difference between simulation steps
            target_speed: desired speed in km/h
            sampling_radius: search radius for next waypoints in m
            lateral_control_dict: dictionary of lateral control parameters
            longitudinal_control_dict: dictionary of longitudinal control parameters
            max_throttle: maximum throttle
            max_brake: maximum brake
            max_steering: maximum steering
            offset: distance between the route waypoints and the center of the lane in m
        :param map_inst: carla.Map instance to avoid the expensive call of getting it
        """
        self._vehicle = vehicle
        self._world = CarlaDataProvider.get_world()
        self._map = self._world.get_map()
        self._local_planner = LocalPlanner(self._vehicle, opt_dict={}, map_inst=self._map)
        self._prev_command = ''

    def set_global_plan(self, plan, stop_waypoint_creation=True, clean_queue=True):
        """
        Adds a specific plan to the agent.

            :param plan: list of [carla.Waypoint, RoadOption] representing the route to be followed
            :param stop_waypoint_creation: stops the automatic random creation of waypoints
            :param clean_queue: resets the current agent's plan
        """
        self._local_planner.set_global_plan(
            plan,
            stop_waypoint_creation=stop_waypoint_creation,
            clean_queue=clean_queue
        )

    def run_step(self, command, atomic_control_step, debug=False):
        """
        Execute one step of local planner and return a control
        """
        command = str(command).lower()
        waypoint = self._map.get_waypoint(self._vehicle.get_location())
        vehicle_pose = self._vehicle.get_transform()
        opposite_lane = False
        if abs(waypoint.transform.rotation.yaw - vehicle_pose.rotation.yaw) > 90 and\
                abs(waypoint.transform.rotation.yaw - vehicle_pose.rotation.yaw) < 270:
            opposite_lane = True

        # # Path Planning
        # if 'change to the' in command and 'change to the' not in self._prev_command and atomic_control_step == 0:
        #     lane_change_direction = 'left' if 'left' in command.lower() else 'right'
        #     opposite_lane_change_direction = 'right' if 'left' in command.lower() else 'left'
        #     print("Lane changing to the left...")
        #     self._plan, self._target_lane_id = generate_target_waypoint_list_multilane(
        #             vehicle_pose, lane_change_direction if not opposite_lane else opposite_lane_change_direction, 2, 2, 8, check=False, lane_changes=1, step_distance=1)
        #     if self._plan:
        #         self._local_planner.set_global_plan(self._plan)
        #     else:
        #         print("no lane change plan possible, following lanes")
        #         self._local_planner.set_global_plan([(self._map.get_waypoint(self._vehicle.get_location()), RoadOption.LANEFOLLOW)])
        #         if opposite_lane:
        #             print("Opposite lane follow")
        #             self._local_planner._compute_next_waypoints(k=-100)
        #         else:
        #             self._local_planner._compute_next_waypoints(k=100)
        #     self._prev_command = command
        #     print("Target lane id: ", self._target_lane_id)
        # elif atomic_control_step == 0:
        #     if not ('change to the' in self._prev_command and len(self._local_planner._waypoints_queue)>1):
        #         self._local_planner.set_global_plan([(self._map.get_waypoint(self._vehicle.get_location()), RoadOption.LANEFOLLOW)])
        #         self._prev_command = command

        #         if 'lane follow' in command:
        #             if opposite_lane:
        #                 print("Opposite lane follow")
        #                 self._local_planner._compute_next_waypoints(k=-100)
        #             else:
        #                 self._local_planner._compute_next_waypoints(k=100)
        #     else:
        #         print("Following lane change plan")

        # Control
        control = self._local_planner.run_step(debug=debug)
        
        if 'speed up' in command:
            control.throttle = 0.75
            control.brake = 0.0
        elif 'slow down' in command:
            control.throttle = 0.0
            control.brake = 0.3
        # elif 'stop' in command:
        #     control.throttle = 0.0
        #     control.brake = 1.0
        elif 'stop' in command:
            control.throttle = 0.0
            control.brake = 0.75
        return control
